/*
	AIPLagClusterDraw_mqK2

	Sampling for latent variables m, q and K.
*/
#define BOUNDS_CHECK 0
#include "mex.h"
#include "matrix.h"
#include "..\C Routines\mxMatrix.h"
#include "..\C Routines\Math2.h"
#include <math.h>
 
inline double floor2(const double a, const double k) { return k*floor(a/k); }
inline double ceil2(const double a, const double k) { return k*ceil(a/k); }
int DrawDiscrete(MatrixWork &Prob, MatrixWork &Outcome, double r);

void mexFunction(int nhls, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	char msg[120];

	if (nrhs!=3) {
		sprintf(msg,"AIPLagClusterDraw_mqK. Expecting 3 inputs; found %d.\n",nrhs);
		mexErrMsgTxt(msg);
	}

	MatrixRhs 
		mOld(prhs[0],"m"), 
		P(prhs[0],"P"), 
		qOld(prhs[0],"q"), 
		x(prhs[0],"x"), 
		ranu(prhs[1]);

	int T = mOld.length;
	if (P.length!=T) {
		sprintf(msg,"AIPLagClusterDraw_mqK. length of p=%d; expecting %d\n", P.length, T);
		mexErrMsgTxt(msg);
	}
	if (qOld.length!=T) {
		sprintf(msg,"AIPLagClusterDraw_mqK. length of q=%d; expecting %d\n", qOld.length, T);
		mexErrMsgTxt(msg);
	}
	if (x.length!=T) {
		sprintf(msg,"AIPLagClusterDraw_mqK. length of x=%d; expecting %d\n", x.length, T);
		mexErrMsgTxt(msg);
	}
	if (ranu.nRows()!=T || ranu.nCols()!=2) {
		sprintf(msg,"AIPLagClusterDraw_mqK. ranu is %dx%d; expecting %dx2\n", ranu.n[0],ranu.n[1], T);
		mexErrMsgTxt(msg);
	}

	MatrixRhs lambda(prhs[2],"lambda");
	//	lambda.Print("lambda");
	int J = lambda.nRows()-1;
	if (lambda.nCols()!=2) {
		sprintf(msg, "AIPLagClusterDraw_mqK. lambda has %d cols; should be 2.\n", lambda.nCols());
		mexErrMsgTxt(msg);
	}
	double
		C			=	GetDoubleFromStruct(prhs[2],"C"),
		varu		=	GetDoubleFromStruct(prhs[2],"varu"),
		ClusterProb	=	GetDoubleFromStruct(prhs[2],"ClusterProb"),
		ClusterMult	=	GetDoubleFromStruct(prhs[2],"ClusterMult");

	int i, j, t, s;
	
	//	New matrices to hold constructed values;
	MatrixLhs m(1,T), q(1,T), K(1,T);

	//	The discrete variable space is q x K = 2 x 2
	//	The row indices correspond to q=-1/+1 and
	//	The column indices correspond to K=1 and K=ClusterMult

	//	mexPrintf("Starting first loop.\n");
	MatrixWork Impact(2,T), u2Sum(2), Outcome(2,2), u(2,T),
		Prob(2,2), Pq(2), PK(2), mu(2), mLower(2,2), mUpper(2,2);

	for (t=1; t<=T; t++) 
	{
		q(t) = qOld(t);
		m(t) = mOld(t);
	}
	
	//	mexPrintf("Starting second loop.\n");
	for (t=1; t<=T; t++)
	{
		//	mexPrintf("t=%d\nSetting PK.\n",t);

		//	Set PK depending on whether P(t) falls on a multiple of ClusterMult
		bool ClusterPossible = (fmod(P(t),ClusterMult)==0);
		if (ClusterPossible) {
			PK(1) = 1-ClusterProb;
			PK(2) = ClusterProb;
		}
		else {
			PK(1) = 1;
			PK(2) = 0;
		}

		//	mexPrintf("Setting mLower and mUpper.\n");
		//	Determine m limits for each q, K pair.
		//	Limits for sells:
		double M = P(t)+C;
		mLower(1,1) = log( M );
		mUpper(1,1) = log( M+1 );
		mLower(1,2) = log( M );
		mUpper(1,2) = log( M+ClusterMult );
		
		//	Limits for buys:
		M = P(t)-C;
		mLower(2,1) = log( M-1 );
		mUpper(2,1) = log( M   );
		mLower(2,2) = log( M-ClusterMult );
		mUpper(2,2) = log( M   );

		double var, sd, ss, u2s;

		//	Recompute the impacts and u's ahead for q(t) = +/- 1
		for (i=1; i<=2; i++)
		{
			if (i==1) q(t)=-1;
			else q(t) = +1;
			for (s=t; s<=__min(t+J,T); s++)
			{
				for (ss=0, j=0; j<=__min(J,s-1); j++) 
					ss += q(s-j)*(lambda(j+1,1)+lambda(j+1,2)*x(s-j));
				Impact(i,s) = ss;
				if (s>1) u(i,s) = m(s) - (m(s-1) + ss);
			}
			u2Sum(i) = 0;
		}

		if (t==1) 
		{
			for (i=1; i<=2; i++)
			{
				if (J>2) {
					for (u2s=0, s=t+2; s<=__min(J+t,T); s++) u2s += sqr(u(i,s));
					u2Sum(i) = u2s;
				}
				Pq(i) = exp(-u2s/(2*varu));
				mu(i) = m(t+1) - Impact(i,t+1);
			}
			Pq.Normalize();
			var = varu;
			//	mexPrintf("Finished mu and Pq for t=1\n");
		}
		else if (t==T) 
		{
			for (i=1; i<=2; i++)
			{
				Pq(i) = .5;
				mu(i) = m(t-1) + Impact(i,t);
			}	
			var = varu;
		}
		else //	1<t<T
		{
			for (i=1; i<=2; i++)
			{
				if (J>1) {
					for (u2s=0, s=t+2; s<=__min(J+t,T); s++) u2s += sqr(u(i,s));
					u2Sum(i) = u2s;
				}
				Pq(i) = exp(-u2Sum(i)/(2*varu) - sqr(u(i,t)+u(i,t+1))/(4*varu));
				mu(i) = (m(t-1) + m(t+1) + Impact(i,t) - Impact(i,t+1))/2;
			}
			Pq.Normalize();
			var = varu/2.;
		}

		sd = sqrt(var);
		for (i=1; i<=2; i++)
			for (j=1; j<=2; j++)
				Prob(i,j) = Pq(i)*PK(j)*
					CDFN(mu(i), sd, mLower(i,j), mUpper(i,j))*exp(-u2Sum(i)/(2*varu));
		Prob.Normalize();
		DrawDiscrete(Prob, Outcome, ranu(t,1));
		//	Outcome.Print("Outcome");

		//	Determine which outcome was drawn:
		for (i=1; i<=2; i++)
			for (j=1; j<=2; j++)
				if (Outcome(i,j)>0) goto DrawDone;

DrawDone:
		if (i==1) q(t) = -1;
		else q(t) = +1;
		if (j==1) K(t) = 1;
		else K(t) = ClusterMult;

		//	Suppress draws of q and K (used for debugging)
		/*
		q(t)=qOld(t);
		if (q(t)==-1) i=1;
		else i=2;
		K(t) = 1;
		j = 1;
		*/
		

		m(t) = RandNormT(mu(i), sd, mLower(i,j), mUpper(i,j), ranu(t,2),100.);

		// Verify validity of generated values
		bool mOkay=true;
		for (int nCheck=0; nCheck<5 && !mOkay; nCheck++) {
			double 
				cbid = exp(m(t))-C,
				bid = floor2(cbid, K(t)),
				cask = exp(m(t))+C,
				ask =  ceil2(cask, K(t));
			if ( (q(t)==-1 && P(t)==bid) || (q(t)==+1 && P(t)==ask)) mOkay=true;
			else {
				mexPrintf("\nError in AsyInfoClusterDraw_mqk at t=%d\tP(t)=%g\n",t,P(t));
				if (q(t)==-1) {
					mexPrintf("Bid side: ");
					mexPrintf("exp(m(t))-C=%25.17g rounded down to %g.\n",cbid,bid);
					double d=cbid-P(t);
					mexPrintf("[exp(m(t))+C]-P(t)=%g\n",d);
					mexPrintf("mLower=%25.17f\n",mLower(i,j));
					mexPrintf("     m=%25.17f\n",m(t));
					if (d<1.e-15) {
						mexPrintf("Raising m(t) by 1e-15.\n");
						m(t) = m(t) + 1e-15;
					}
				}
				else {
					mexPrintf("Ask side: ");
					mexPrintf("exp(m(t))+C=%25.17g rounded up to %g.\n",cask,ask);
					double d=P(t)-cask;
					mexPrintf("P(t)-[exp(m(t))+C]=%g\n",d);
					mexPrintf("     m=%25.17f\n",m(t));
					mexPrintf("mUpper=%25.17f\n",mUpper(i,j));
					if (d<1.e-15) {
						mexPrintf("Lowering m(t) by 1e-15.\n");
						m(t) = m(t) - 1e-15;
					}
				}
			}
		}
		if (!mOkay) mexErrMsgTxt("Terminating.");
	}

	//	Create return structure
	int dims[]={1,1};
	const char *fn[]={"m","q","K"};
	plhs[0]=mxCreateStructArray(2, dims, 3, fn);
	if (plhs[0]==NULL) mexPrintf("Error in creating return structure");
	mxSetField(plhs[0], 0, "m", m.mx);
	mxSetField(plhs[0], 0, "q", q.mx);
	mxSetField(plhs[0], 0, "K", K.mx);
//	mexErrMsgTxt("That's all.");
}

//	Make a random draw from a discrete probability distribution.
//	In the calling program, Outcome should be dimensioned exactly as Prob.
//	On return, the element of Outcome corresponding to the drawn outcome is set to one.
//	Within the routine, Prob and Outcome are treated as vectors.
int DrawDiscrete(MatrixWork &Prob, MatrixWork &Outcome, double r)
{
	int nState = Prob.length;
	for (int i=1; i<=nState; i++) Outcome(i)=0;
	double CumProb = 0;
	for (i=1; i<=nState; i++)
	{
		double CumProbNext = CumProb + Prob(i);
		if (r>CumProb && r<=CumProbNext)
		{
			Outcome(i) = 1;
			return 0;
		}
		CumProb = CumProbNext;
	}
	mexErrMsgTxt("DiscreteDraw error.\n");
	return 1;
}
